import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;

import java.io.*;

/**
 * A simple sound player. To start, create an instance of this class.
 * 
 * The sound player provides an interface to the MusicOrganizer class
 * from chapter 4.
 * 
 * @author Michael Kölling and David J. Barnes
 * @version 1.0
 */
public class MusicPlayerGUI extends JFrame
    implements ChangeListener, ActionListener
{
    private static final String VERSION = "Version 1.0";
    private static final String AUDIO_DIR = "audio";
    
    private JList fileList;
    private JSlider slider;
    private JLabel infoLabel;
    private JComboBox formatList;
    private MusicOrganizer organizer;
    // A player for the music tracks.
    private MusicPlayer player;
    // The current track list.
    private List<Track> trackList;
    // Whether anything is currently playing.
    private boolean playing;
    // A thread to track the current playing position,
    // for the slider.
    private Thread playingTracker;

    /**
     * Main method for starting the player from a command line.
     */
    public static void main(String[] args)
    {
        MusicPlayerGUI gui = new MusicPlayerGUI();
    }
    
    /**
     * Create a SoundPlayer and display its GUI on screen.
     */
    public MusicPlayerGUI()
    {
        super("Music Player");
        organizer = new MusicOrganizer(AUDIO_DIR);
        player = new MusicPlayer();
        playing = false;
        
        makeFrame();
        setupTracker();
    }
    
    /**
     * Play the sound file currently selected in the file list. If there is no
     * selection in the list, or if the selected file is not a sound file, 
     * do nothing.
     */
    private void play()
    {
        // Make sure there is nothing playing at the moment.
        stop();
        int index = fileList.getSelectedIndex();
        if(index >= 0 && index < trackList.size()) {
            Track tk = trackList.get(index);
            showInfo("Now playing: " + tk.getTitle() + " by " + tk.getArtist());
            player.startPlaying(tk.getFilename());
            setSliderLabels(player.getLength());
            playing = true;
        }
    }
    
    /**
     * Stop the currently playing sound file (if there is one playing).
     */
    private void stop()
    {
        playing = false;
        showInfo("");
        player.stop();
    }

    /**
     * Stop the currently playing sound file (if there is one playing).
     */
    private void pause()
    {
        player.pause();
    }

    /**
     * Resume a previously suspended sound file.
     */
    private void resume()
    {
        player.resume();
    }
    
    /**
     * Track the current playing position, which a track
     * is being played.
     */
    private void setupTracker()
    {
        playingTracker = new Thread(new Runnable(){
            public void run()
            {
                slider.setValue(0);
                while(true) {
                    int position;
                    long sleepTime;
                    if(playing) {
                        position = player.getPosition();
                        sleepTime = 200;
                    }
                    else {
                        position = 0;
                        sleepTime = 1000;
                    }
                    slider.setValue(position);
                    try {
                        Thread.sleep(sleepTime);
                    }
                    catch(InterruptedException e) {
                    }
                }
            }
        });
        playingTracker.start();
    }
    
    /**
     * Reload the library of tracks.
     */
    private void reloadTrackList()
    {
        organizer.clearTracks();
        organizer.readLibrary(AUDIO_DIR);
        int index = formatList.getSelectedIndex();
        if(index < 0) {
            index = 0;
        }
        setListOrdering((String) formatList.getItemAt(0));
    }

    /**
     * Display information about a selected sound file (name and clip length).
     * @param message The message to display.
     */
    private void showInfo(String message)
    {
        infoLabel.setText(message);
    }
    
    /**
     * Quit function: quit the application.
     */
    private void quit()
    {
        System.exit(0);
    }
    
    
    /**
     * About function: show the 'about' box.
     */
    private void showAbout()
    {
        JOptionPane.showMessageDialog(this, 
                    "Music Player\n" + VERSION,
                    "About Music Player", 
                    JOptionPane.INFORMATION_MESSAGE);
    }
    
    // ------- ChangeListener interface (for Slider) -------

    /**
     * ChangeListener method for slider changes. This method is called
     * when the slider value is changed by the user.
     * @param evt The event details.
     */
    public void stateChanged(ChangeEvent evt)
    {
        // Seek to a new position in the current file.
        player.seekTo(slider.getValue());
    }
    
    // ------- ActionListener interface (for ComboBox) -------

    /**
     * ActionListener method for changes of format combo box.
     * When this method is called, the user has made a new selection 
     * in the combo box.
     * @param evt The event details.
     */
    public void actionPerformed(ActionEvent evt) 
    {
        JComboBox cb = (JComboBox) evt.getSource();
        int index = cb.getSelectedIndex();
        if(index >= 0) {
            setListOrdering((String) cb.getItemAt(index));
        }
    }
    
    /**
     * Set the ordering of the track list.
     * @param ordering The ordering to use.
     */
    private void setListOrdering(String ordering)
    {
        trackList = organizer.sortByField(ordering);
        String[] tracks = getTracksDisplayList(trackList);
        fileList.setListData(tracks);
    }

    /**
     * Get a display version of the track list.
     * @param trackList The list of tracks to be displayed.
     * @return The tracks in display format.
     */
    private String[] getTracksDisplayList(List<Track> trackList)
    {
        int numTracks = trackList.size();
        String[] tracks = new String[numTracks];
        for(int i = 0; i < numTracks; i++) {
            String[] fields = trackList.get(i).getFields();
            StringBuilder listing = new StringBuilder();
            for(String field : fields) {
                listing.append(field);
                listing.append(" ");
            }
            tracks[i] = listing.toString().trim();
        }
        return tracks;
    }

    /**
     * Set up the labels on the slider, based on the length of
     * the track being played.
     * @param trackLength The length of the track (in arbitrary units).
     */
    private void setSliderLabels(int trackLength)
    {
        if(trackLength > 0) {
            // How many ticks we want.
            int numberOfTicks = 10;
            // Place ticks to the nearest 1000.
            int roundedLength = trackLength - (trackLength % 1000);
            if(roundedLength == 0) {
                roundedLength = trackLength;
            }
            slider.setMajorTickSpacing(roundedLength / numberOfTicks);
            slider.setMinorTickSpacing(0);
            slider.setPaintLabels(true);
            slider.setMaximum(trackLength);
        }
    }

    // ---- Swing stuff to build the frame and all its components and menus ----
    
    /**
     * Create the complete application GUI.
     */
    private void makeFrame()
    {
        // the following makes sure that our application exits when
        // the user closes its window
        setDefaultCloseOperation(EXIT_ON_CLOSE);
        
        JPanel contentPane = (JPanel) getContentPane();
        contentPane.setBorder(new EmptyBorder(6, 10, 10, 10));

        makeMenuBar();
        
        // Specify the layout manager with nice spacing
        contentPane.setLayout(new BorderLayout(8, 8));

        // Create the left side with combobox and scroll list
        JPanel leftPane = new JPanel();
        {
            leftPane.setLayout(new BorderLayout(8, 8));

            // Set up components for ordering the list of tracks.
            JPanel orderingPanel = new JPanel();
            orderingPanel.setLayout(new BorderLayout());
            orderingPanel.add(new JLabel("Order by:"), BorderLayout.NORTH);
            
            // Get the list of field names, used for ordering.
            String[] ordering = Track.FIELDS;
            
            // Create the combo box.
            formatList = new JComboBox(ordering);
            formatList.addActionListener(this);
            orderingPanel.add(formatList, BorderLayout.CENTER);
            
            leftPane.add(orderingPanel, BorderLayout.NORTH);
    
            // Create the scrolled list for track listing.
            fileList = new JList();
            fileList.setForeground(new Color(140,171,226));
            fileList.setBackground(new Color(0,0,0));
            fileList.setSelectionBackground(new Color(87,49,134));
            fileList.setSelectionForeground(new Color(140,171,226));
            
            // Have double-clicks on the list call play().
            fileList.addMouseListener(new MouseAdapter() {
             public void mouseClicked(MouseEvent e) {
                 if (e.getClickCount() == 2) {
                     play();
                  }
             }
            });
            JScrollPane scrollPane = new JScrollPane(fileList);
            scrollPane.setColumnHeaderView(new JLabel("Audio files"));
            leftPane.add(scrollPane, BorderLayout.CENTER);
            
            // Set up the initial listing.
            setListOrdering(ordering[0]);
        }
        contentPane.add(leftPane, BorderLayout.CENTER);

        // Create the center with image, text label, and slider
       JPanel centerPane = new JPanel();
        {
            centerPane.setLayout(new BorderLayout(8, 8));
    
            JLabel image = new JLabel(new ImageIcon("title.jpg"));
            centerPane.add(image, BorderLayout.NORTH);
            centerPane.setBackground(Color.BLACK);

            infoLabel = new JLabel("abc");
            infoLabel.setHorizontalAlignment(SwingConstants.CENTER);
            infoLabel.setForeground(new Color(140,171,226));
            centerPane.add(infoLabel, BorderLayout.CENTER);

            slider = new JSlider(0, 100, 0);
            TitledBorder border = new TitledBorder("Seek");
            border.setTitleColor(Color.white);
            slider.setBorder(new CompoundBorder(new EmptyBorder(6, 10, 10, 10), border));
            slider.addChangeListener(this);
            slider.setBackground(Color.BLACK);
            //slider.setMajorTickSpacing(25);
            slider.setPaintTicks(true);
            slider.setPaintLabels(true);
            centerPane.add(slider, BorderLayout.SOUTH);
        }
        contentPane.add(centerPane, BorderLayout.EAST);

        // Create the toolbar with the buttons
        JPanel toolbar = new JPanel();
        {
            toolbar.setLayout(new GridLayout(1, 0));
  
            JButton button = new JButton("Play");
            button.addActionListener(new ActionListener() {
                       public void actionPerformed(ActionEvent e) {
                           play();
                       }
                   });
            toolbar.add(button);
            
            button = new JButton("Stop");
            button.addActionListener(new ActionListener() {
                       public void actionPerformed(ActionEvent e) { 
                           stop(); 
                       }
                   });
            toolbar.add(button);
    
            button = new JButton("Pause");
            button.addActionListener(new ActionListener() {
                       public void actionPerformed(ActionEvent e) { 
                           pause(); 
                       }
                   });
            toolbar.add(button);
            
            button = new JButton("Resume");
            button.addActionListener(new ActionListener() {
                       public void actionPerformed(ActionEvent e) { 
                           resume(); 
                       }
                   });
            toolbar.add(button);

        }
        
        contentPane.add(toolbar, BorderLayout.NORTH);
        
        // building is done - arrange the components      
        pack();
        
        // place this frame at the center of the screen and show
        Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
        setLocation(d.width/2 - getWidth()/2, d.height/2 - getHeight()/2);
        setVisible(true);
    }
    
    /**
     * Create the main frame's menu bar.
     */
    private void makeMenuBar()
    {
        final int SHORTCUT_MASK =
            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();

        JMenuBar menubar = new JMenuBar();
        setJMenuBar(menubar);
        
        JMenu menu;
        JMenuItem item;
        
        // create the File menu
        menu = new JMenu("File");
        menubar.add(menu);
        
        item = new JMenuItem("Reload");
            item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_R, SHORTCUT_MASK));
            item.addActionListener(new ActionListener() {
                   public void actionPerformed(ActionEvent e) { 
                       reloadTrackList(); 
                   }
               });
        menu.add(item);
        
        item = new JMenuItem("Quit");
            item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, SHORTCUT_MASK));
            item.addActionListener(new ActionListener() {
                   public void actionPerformed(ActionEvent e) { 
                       quit(); 
                   }
               });
        menu.add(item);

        // create the Help menu
        menu = new JMenu("Help");
        menubar.add(menu);
        
        item = new JMenuItem("About SoundPlayer...");
            item.addActionListener(new ActionListener() {
                   public void actionPerformed(ActionEvent e) { 
                       showAbout(); 
                   }
               });
        menu.add(item);
    }
}
